//////////////////////////////////////////////////////////////////////////////////////
//
//  Copyright (c) 2014-present, Egret Technology.
//  All rights reserved.
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of the Egret nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY EGRET AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
//  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL EGRET AND CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE, DATA,
//  OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
//  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
//  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//////////////////////////////////////////////////////////////////////////////////////

interface BatchETC1FileWriter {
    name: string,               //文件名
    head: number,               //留着，随便干点什么, 可以填个版本号什么的
    size: number[],             //[colorTextureBuffer size][alphaMaskTextureBuffer size]
    buffers: ArrayBuffer[],     //[colorTextureBuffer arraybuffer][alphaMaskTextureBuffer arraybuffer]
    writePath: string           //写文件的路径
}

export class PackETC1KTXColorAndAlphaMaskPlugin implements plugins.Command {

    private readonly etc1color: plugins.File[] = [];
    private readonly etc1alpha: plugins.File[] = [];
    private readonly etc1store: { [index: string]: plugins.File[] } = {};
    private readonly batchETC1FileWriters: BatchETC1FileWriter[] = [];
    private readonly debugLog: boolean = false;

    constructor(debugLog: boolean) {
        this.debugLog = debugLog;
    }

    async onFile(file: plugins.File) {
        const extname = file.extname;
        if (extname === '.ktx') {
            const basename = file.basename;
            if (basename.indexOf('.pvr.ktx') >= 0) {
                //这个格式不在本插件中处理
            }
            else if (basename.indexOf('.pack.etc1.ktx') <= 0) {
                //不再处理.pack.etc1.ktx
                if (basename.indexOf('_alpha.ktx') >= 0) {
                    //是alpha遮罩
                    this.etc1alpha.push(file);
                    return null;
                }
                else if (basename.indexOf('.ktx') >= 0) {
                    //是颜色
                    this.etc1color.push(file);
                    return null;
                }
            }
        }
        return file;
    }

    private makeEtc1Store(): void {
        //
        const etc1store = this.etc1store;
        //采集颜色的文件
        const etc1color = this.etc1color;
        for (const file of etc1color) {
            const basename = file.basename;
            const index = basename.indexOf('.ktx');
            if (index >= 0) {
                const name = basename.substr(0, index);
                const info = etc1store[name] = etc1store[name] || [];
                info[0] = file;
            }
        }
        //采集alpha遮罩的文件
        const etc1alpha = this.etc1alpha;
        for (const file of etc1alpha) {
            const basename = file.basename;
            const index = basename.indexOf('_alpha.ktx');
            if (index >= 0) {
                const name = basename.substr(0, index);
                const info = etc1store[name] = etc1store[name] || [];
                info[1] = file;
            }
        }
        //调试打印看结果
        if (this.debugLog) {
            for (const key in etc1store) {
                const info = etc1store[key];
                console.log('___________etc1store key = ' + key);
                console.log('color = ' + info[0].path);
                console.log('alpha = ' + (info[1] ? info[1].path : 'null'));
            }
        }
    }

    private makeBatchETC1FileWriters(): void {
        const etc1store = this.etc1store;
        const debugLog = this.debugLog;
        for (const key in etc1store) {
            const store = etc1store[key];
            if (!store[0]) {
                if (debugLog) {
                    console.warn('makeBatchETC1FileWriters lose color texture, key = ' + key);
                }
                continue;
            }
            /*
            * 4(head)-4(color buf size, size[0])-4(alpha buf size, size[1])-data1(color buf data, buffer[0])-data2(alpha buf data, buffer[1])
            */
            //
            const writer = {} as BatchETC1FileWriter;
            writer.name = key;
            writer.head = 0;
            //
            const size = writer.size = writer.size || [0, 0];
            const buffers = writer.buffers = writer.buffers || [new ArrayBuffer(0), new ArrayBuffer(0)];
            //
            const colorFile = store[0];
            size[0] = colorFile.contents.byteLength;
            buffers[0] = colorFile.contents.buffer;
            //
            const alphaFile = store[1];
            if (alphaFile) {
                size[1] = alphaFile.contents.byteLength;
                buffers[1] = alphaFile.contents.buffer;
            }
            else {
                //是没有alpha遮罩的图
            }
            //
            const relativeIndex = colorFile.relative.indexOf(colorFile.basename);
            if (relativeIndex >= 0) {
                writer.writePath = colorFile.relative.substr(0, relativeIndex);
            }
            else {
                console.error('colorFile.relative = ' + colorFile.basename);
                console.error('colorFile.relative = ' + colorFile.relative);
                continue; //无法写入
            }
            //添加1个新的writeer
            this.batchETC1FileWriters.push(writer);
        }
    }

    private writeFiles(commandContext: plugins.CommandContext): void {
        if (!commandContext) {
            return;
        }
        const batchETC1FileWriters = this.batchETC1FileWriters;
        const debugLog = this.debugLog;
        for (const writer of batchETC1FileWriters) {

            //计算总大小，并new数据
            let writeSize = /*head*/4 + /*color size*/4 + /*alpha size*/4;
            writeSize += /*color buffer*/writer.buffers[0].byteLength;
            writeSize += /*alpha buffer*/writer.buffers[1].byteLength;
            const data = new ArrayBuffer(writeSize);

            //写head, size x 2;
            const u32View = new Uint32Array(data);
            //write head
            u32View[0] = writer.head;
            //write size 
            u32View[1] = writer.size[0];
            u32View[2] = writer.size[1];
            let offset = /*type*/4 + /*color size*/4 + /*alpha size*/4;

            //write buffers
            const u8View = new Uint8Array(data);
            //write color buffer
            u8View.set(new Uint8Array(writer.buffers[0]), offset);
            //color buffer offset size
            offset += writer.size[0];
            //write alpha buffer
            u8View.set(new Uint8Array(writer.buffers[1]), offset);

            //
            const writePath = writer.writePath + writer.name + '.pack.etc1.ktx';
            if (debugLog) {
                console.log('write = ' + writePath);
            }
            commandContext.createFile(writePath, new Buffer(data));
        }
    }

    async onFinish(commandContext: plugins.CommandContext) {
        //收集数据
        console.log('makeEtc1Store______________________');
        this.makeEtc1Store();
        //构建写入文件
        console.log('makeBatchETC1FileWriters______________________');
        this.makeBatchETC1FileWriters();
        //开始写
        console.log('writeFiles______________________');
        this.writeFiles(commandContext);
    }
}
